/*  

sample.c : A sample file illustrating the use of PGSL library.
Dejong's first function F1 is used as the objective function.

Copyright (C) Benny Raphael  

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or  any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.


*/


#include <stdio.h>
#include <math.h>

/*  Including the PGSL header file  */
#include "PGSL.h"

/* The number of variables in the present problem  */
#define numVars 3

/* To keep track of the number of evaluations of the objective function  */
int numTrials = 0;

/* The objective function for DeJong's first function F1

DeJong's first function is:
//
// f1(x1,x2,x3) = x1*x1 + x2*x2 + x3*x3
//
// where each x is in the range [-5.12,5.12]
*/
double objectiveFunction(struct ProblemSetup *setup, double *paramValues){

  double value=0;

  /* Getting the values of parameters from the array.  */
  double x1 = paramValues[0];
  double x2 = paramValues[1];
  double x3 = paramValues[2];

  /*  Calculating the value of the objective function  */
  value = x1*x1 + x2*x2 + x3*x3;

  /*  Updating the number of evaluations  */
  numTrials++;

  printf("\t%d\t%8.3lf\t%8.3lf\t%8.3lf\t%8.3lf\n", numTrials, x1, x2, x3, value); /*  */

  return value;
}

main() {

   /*  Structure representing the optimisation problem  */
   struct ProblemSetup *setup;
    
   int i;

   /*  The minimum values of variables  */
   double  min[] = { -5.12, -5.12, -5.12 };

   /*  The maximum values of variables  */
   double max[] = { 5.12, 5.12, 5.12 };

   /* The number of focusing cycles  */
   int NFC = 60;

   /*  The number of subdomain cycles  */
   int NSDC = 20;

   /*  The random seed value.  This should be changed for multiple runs  */
   int randomseed = 0;

   /*  The minimum value of the objective function at which the algorithm
	should terminate.  An impossibly small value could be given if 
	you have no idea of the minimum.
   */
   double threshold = 1e-8;

   /*  The array to store the values of parameters for the minimum point */
   double minimumpoint[numVars];

   /*  The actual minimum value of the objective function found  by PGSL  */
   double minvalue;

   /*  Setting up the problem */
   setup = ProblemSetup_create(numVars, NFC, NSDC, threshold);
   setup->costFunction = objectiveFunction;
   for (i=0; i<3; i++) {
       setup->axes[i] = PAxis_create(min[i],  max[i]);
	   setup->axes[i]->axisPrecision = 1e-5;
   }
   setup->suppressOutput = 2;


   /* Calling PGSL  */
   minvalue = findMinimum(setup);

   /*  Printing the results */
   printf("\nResults\n");
   printf("\t x1 = %g\n", setup->minimumPoint[0]);
   printf("\t x2 = %g\n", setup->minimumPoint[1]);
   printf("\t x3 = %g\n", setup->minimumPoint[2]);
   printf("\t objective function = %g\n", minvalue);


   printf("%ld Trials\n", numTrials);
}


