/*  

sample.c : A sample file illustrating the use of PGSL library.
Dejong's first function F1 is used as the objective function.

Copyright (C) Benny Raphael  

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or  any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.


*/


#include <stdio.h>
#include <math.h>

/*  Including the PGSL header file  */
#include "..\PGSL.h"

/* The number of variables in the present problem  */
int numVars = 4;

/* To keep track of the number of evaluations of the objective function  */
int numTrials = 0;

double a[4][10] = 
	{	{4, 1, 8, 6, 3, 2, 5, 8, 6, 7}, 
		{4, 1, 8, 6, 7, 9, 5, 1, 2, 3.6 }, 
		{4, 1, 8, 6, 3, 2, 3, 8, 6, 7 }, 
		{4, 1, 8, 6, 7, 9, 3, 1, 2, 3.6 } };
double c[10] = {0.1, 0.2, 0.2, 0.4, 0.4, 0.6, 0.3, 0.7, 0.5, 0.5 };

int shekelInstance=5;		/*  The instance of the shekel function 5, 7, ro 10  */

/* Known solutions to shekel index starts from 0   */
double knownSolutions[11] = {0, 0, 0, 0, 0, -10.1531996790582, 0, -10.4029405668187, 0, 0, -10.5364098166920 };		

double bestcost = 1e100;

#define NUMRUNS 25

/* The objective function 

*/
double objectiveFunction(struct ProblemSetup *setup, double *paramValues){

  double value=0;
  int m;
  double *x = paramValues;
  double sum;
  int i,j;

  m = shekelInstance;
  value = 0;
  for (i=0; i<m; i++) {
	sum = 0;
	for (j=0; j< numVars; j++)  sum += (x[j] - a[j][i])*(x[j] - a[j][i]);
	sum += c[i];
	value += 1.0/sum;
  }
  value = -value;

  /*  Updating the number of evaluations  */
  numTrials++;

  
  if (value < bestcost) {
	  bestcost = value;
	  /* printf("%d\t%lf\n", numTrials, bestcost); */
  }

  return value;
}

main() {

   /*  Structure representing the optimisation problem  */
   struct ProblemSetup *setup;
    
   int i;

   /*  The minimum values of variables  */
   double  min[] = { 0, 0, 0, 0 };

   /*  The maximum values of variables  */
   double max[] = { 10, 10, 10, 10 };

   /* The number of focusing cycles  */
   int NFC = 40;

   /*  The number of subdomain cycles  */
   int NSDC = 20;

   /*  The random seed value.  This should be changed for multiple runs  */
   int randomseed = 0;

   /*  The minimum value of the objective function at which the algorithm
	should terminate.  An impossibly small value could be given if 
	you have no idea of the minimum.
   */
   double threshold;

   /*  The actual minimum value of the objective function found  by PGSL  */
   double minvalue;

   int instance;
   int run;
   int evals[NUMRUNS];
   double costs[NUMRUNS];

   int numSuccess=0;
   double sumValue=0;
   long sumTrials=0;
   double meanTrials;
   double meanValue;
   FILE *out;
   char filename[256];


   for (instance=0; instance < 3; instance++) {

	   numSuccess=0;
	   sumValue=0;
	   sumTrials=0;


	   if (instance == 0) shekelInstance=5;
	   else if (instance == 1) shekelInstance=7;
	   else shekelInstance=10;
		
	   threshold = knownSolutions[shekelInstance] + 0.01*fabs(shekelInstance);

		NFC = 20*numVars;
		NSDC = 300/numVars;

	   for (run=0; run < NUMRUNS; run++) {
			/*  Setting up the problem */
			setup = ProblemSetup_create(numVars, NFC, NSDC, threshold);
			setup->randomSeed = run;
			setup->costFunction = objectiveFunction;
			for (i=0; i<numVars; i++) {
				setup->axes[i] = PAxis_create(min[i],  max[i]);
				setup->axes[i]->axisPrecision = 0.01;
			}


			/* Calling PGSL  */
			minvalue = findMinimum(setup);

			if (setup->globalMinimum < setup->threshold) {
				sumTrials += numTrials;
				numSuccess++;
			}
			sumValue += setup->globalMinimum;

			evals[run] = numTrials;
			costs[run] = minvalue;
			numTrials = 0;
			ProblemSetup_delete(setup);
		}

	    printf("Results for shekel instance: %d\n", shekelInstance);
		for (run =0; run <25; run++) {
			printf ("\t%d\t%d\t%lf\n", run, evals[run], costs[run]);
		}

		if (numSuccess > 0) {
			meanTrials = sumTrials/numSuccess;
		} else {
			meanTrials = evals[0];
		}
		meanValue = sumValue/NUMRUNS;

		sprintf(filename, "Shekel-%d.out", shekelInstance);
		out = fopen(filename, "w");
		if (out == NULL) return 1;

		fprintf(out, "Shekel-%d\n", shekelInstance);
		fprintf(out, "Seed \t Evaluations \t Minimum found \n");
		for (run =0; run <NUMRUNS; run++) {
			fprintf(out, "\t%d\t%ld\t%lg\n", run, evals[run], costs[run]);
		}
		fprintf(out, "\n");
		fprintf(out, "Mean number of evaluations = %ld\n", (long)meanTrials);
		fprintf(out, "Mean Value of minima = %lg\n", meanValue);
		fprintf(out, "Success rate = %d %% \n", 100*numSuccess/NUMRUNS);

		fclose(out);


	}
	return 0;

}


