/*  
Optimser.c - Internal routines of PGSL, version 2.0 Beta

Copyright (C) Benny Raphael  

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or  any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.


*/

#include <stdio.h>
#include <malloc.h>
#include <stdlib.h>
#include <math.h>
#include <string.h>

#include  "PGSL.h"

/** <A HREF=Optimiser.c> Code </A> <P>
  * 
  * @version        0.0 Tue Aug 24 16:32:57 PDT 1999 
  * @author         Benny Raphael 

*/
#define defaultScaleFactor 0.35
#define defaultAxisPrecision 1e-8


static int debug = 0;
static int testing = 20;

static int DefaultNumIterations = 20;
static int DefaultNumPointsPerIteration = 100;
static int DefaultNumGenerations = 20;
static int DefaultNumSVC = 2;
static double widenFactor = 1e-1;

/*  The maximum number of solutions that are saved for computing statistics of convergence  */
int numSavedSolutions=5;


/*  Function to create an axis representing a parameter.  */
struct PAxis *PAxis_create(double min, double max) {
   struct PAxis *ret;
   int i;
   double dx, p;
   int numintervals=20;

   ret = (struct PAxis *)calloc(1, sizeof(struct PAxis) );
   if (ret == NULL) memory_error();

   ret->type = 1;
   ret->min = min;
   ret->max = max;
   ret->numIntervals = numintervals;

   if (numintervals == 0) return ret;
   ret->intervals = (double *)calloc(numintervals,sizeof(double) );
   ret->prob = (double *)calloc(numintervals,sizeof(double) );
   ret->cdf = (double *)calloc(numintervals,sizeof(double) );
   if (ret->intervals == NULL  || ret->prob == NULL  
	|| ret->cdf == NULL  ) memory_error();

   dx = (max-min)/numintervals;
   p = 1.0/numintervals;
   for (i=0; i <numintervals; i++) {
      ret->intervals[i] = min + i*dx;
      ret->prob[i] = p;
      ret->cdf[i] = i*p;
   }

   ret->axisPrecision = defaultAxisPrecision;

   return ret;
}
/*----------------------------------------------------------------*/

/*  Frees memory after checking the validity of object.  If the type is 
   less than 0 it does not delete  */
int PAxis_delete(struct PAxis *thisp) {
   if (thisp == NULL) return 0;
   if (thisp->type < 0) return 0;  /* Precaution  */
   if (thisp->numIntervals == 0) return 0;
   if (thisp->intervals != NULL) free(thisp->intervals);
   if (thisp->prob != NULL) free(thisp->prob);
   if (thisp->cdf != NULL) free(thisp->cdf);
   thisp->type = -1;/* Precaution  */
   free(thisp);
   return 1;
}
/*----------------------------------------------------------------*/


double PAxis_valueForCDF(struct PAxis *thisp, double t) {
   double x1, x2, ret, t1, dt;
   int i;


   for (i=0; i<thisp->numIntervals; i++) {
      if (t < thisp->cdf[i]) break;
   }
   if (i==0) return thisp->intervals[0];
   t1 = thisp->cdf[i-1];
   dt = t - t1;
   x1 = thisp->intervals[i-1];
   if (i == thisp->numIntervals) x2 = thisp->max;
   else x2 = thisp->intervals[i];
   ret = x1 + dt*(x2 - x1);

   return ret;
}
/*---------------------------------------------------------------------------*/

int PAxis_indexOf(struct PAxis *thisp,double x){
   int i;
   for (i=0; i<thisp->numIntervals; i++) {
      if (x < thisp->intervals[i]) break;
   }
   if (i==0) return 0;
   return i-1;
}
/*---------------------------------------------------------------------------*/

int copyPoint(int n, const double *x, double *copy) {
	memcpy(copy, x, n*sizeof(double) );
	return 1;
}
/*----------------------------------------------------------------*/

/*---------------------------------------------------------------------------*/
/*  Function to create the structure representing a problem definition  */
struct ProblemSetup *ProblemSetup_create(int num, int NFC,  long NSDC, double threshold ) {
   struct ProblemSetup *ret = (struct ProblemSetup *)calloc(1,
	 sizeof(struct ProblemSetup) );
   if (ret == NULL) memory_error();
   ret->numVars=num;
   ret->NS = 2;
   ret->NPUC=1;
   ret->NFC=NFC;
   ret->NSDC = NSDC;
   ret->axes = (struct PAxis **)calloc(num, sizeof(struct PAxis *) );
   ret->minimumPoint = (double *)calloc(num, sizeof(double) );
   ret->SDCpoints = (double *)calloc(ret->numVars*numSavedSolutions, sizeof(double) );
   ret->SDCcosts = (double *)calloc(numSavedSolutions, sizeof(double) );


   if (ret->axes == NULL || ret->minimumPoint == NULL ||
    ret->SDCpoints  == NULL || ret->SDCcosts == NULL ) memory_error();

   ret->scaleFactor = defaultScaleFactor;
   ret->suppressOutput = 2;
   ret->iSDC=0;
   ret->userData=NULL;
   ret->threshold = threshold;
   ret->restart = 0;
   ret->adaptive=1;

   return ret;
}
/*----------------------------------------------------------------*/

/*  Function to delete a ProblemSetup.  Frees memory only after validating
thisp object.  If numVars < 0 does not free memeoyr  */
int ProblemSetup_delete(struct ProblemSetup *thisp) {
   int i;

   if (thisp == NULL) return 0;
   if (thisp->numVars < 0) return 0;

   for (i=0; i<thisp->numVars; i++)
      if (thisp->axes[i] != NULL) PAxis_delete(thisp->axes[i]);
   if (thisp->axes != NULL) free(thisp->axes);
   if (thisp->minimumPoint != NULL) free(thisp->minimumPoint);
   if (thisp->SDCpoints != NULL) free(thisp->SDCpoints);
   if (thisp->SDCcosts != NULL) free(thisp->SDCcosts);
   if (thisp->backupMinimumPoint != NULL) free(thisp->backupMinimumPoint);

   thisp->numVars = -1;
   free(thisp);
   return 1;
}
/*----------------------------------------------------------------*/


int ProblemSetup_updateMinimum(struct ProblemSetup *thisp, double *point, double cost){

   if (thisp->foundmin == 0) {
      thisp->foundmin=1;
      thisp->globalMinimum = cost;
      copyPoint(thisp->numVars, point, thisp->minimumPoint);
      return 1;
   } 
   if (cost < thisp->globalMinimum) {
      thisp->globalMinimum = cost;
      copyPoint(thisp->numVars, point, thisp->minimumPoint);
   } 

   return 1;
}
/*---------------------------------------------------------------------------*/


int updateProbability(struct ProblemSetup *setup) {
   int i, j, index;
   double sum, pf1, pf2; /* increase in probability  */

   pf1 = 1.2;
   pf2 = 1.19;
    

   for (i=0; i<setup->numVars; i++) {
      index = PAxis_indexOf(setup->axes[i], setup->minimumPoint[i]);
      if (index != 0) setup->axes[i]->prob[index-1] *= pf2;
      setup->axes[i]->prob[index] *= pf1;
      if (index != setup->axes[i]->numIntervals-1) 
	setup->axes[i]->prob[index+1] *= pf2;
      sum = 0;
      for (j=0; j<setup->axes[i]->numIntervals; j++) 
	sum += setup->axes[i]->prob[j];
      if (sum == 0) continue;
      for (j=0; j<setup->axes[i]->numIntervals; j++) {
	setup->axes[i]->prob[j] /= sum;
	if (j != setup->axes[i]->numIntervals-1) 
	   setup->axes[i]->cdf[j+1] =  setup->axes[i]->cdf[j] + setup->axes[i]->prob[j];
      }

      
      if (!setup->suppressOutput) printf("Update axis %d\n", i);
      for (j=0; j<setup->axes[i]->numIntervals; j++) {
	float div;
	if (j != setup->axes[i]->numIntervals-1) 
	   div = (float)( setup->axes[i]->intervals[j+1] - setup->axes[i]->intervals[j] );
	else div = 1;
	if (!setup->suppressOutput) 
	   printf("\t%g\t%g\n", setup->axes[i]->intervals[j],  setup->axes[i]->prob[j]/div);
      }

   }
   
   return 1;
}
/*---------------------------------------------------------------------------*/

/*  It divides the specified interval into 6.  intervals before and after index
are collapsed so that the resulting total interval is still 20.  All intervals
are assigned equal probability so that effectively the probability within index
is multiplied 10 fold  */
int divideInterval(struct PAxis *axis, int index, double minimumPoint, double mindx, double amin, double amax) {

   double x1, x2, dx, dx1, dx2, p, pind, p1, p2, sum, f1, f2, xc;
   int i, n1, n2, nd=6, nr;
   nr = axis->numIntervals-nd;

   /*  Added  Mar 3 2002 */
   if (minimumPoint < axis->min) {
		if (minimumPoint < amin) minimumPoint = amin;
		axis->min = minimumPoint;
		dx = axis->intervals[1] - axis->intervals[0];
		axis->intervals[0] = axis->min;
		axis->intervals[1] = (axis->intervals[0] + axis->intervals[1])/2.;
/* printf("%d\t%lg\t%lg\t%lg\n",index, minimumPoint, axis->intervals[1],axis->intervals[2]); */
   }
   if (minimumPoint > axis->max){
		if (minimumPoint > amax) minimumPoint = amax;
		dx = axis->max - axis->intervals[axis->numIntervals-1];
		axis->max = minimumPoint;
		axis->intervals[axis->numIntervals-1] = (axis->max + axis->intervals[axis->numIntervals-1])/2.;
/* printf("%d\t%lg\t%lg\t%lg\n",index, axis->intervals[axis->numIntervals-2],axis->intervals[axis->numIntervals-1],minimumPoint); */
   }

   x1 = axis->intervals[index];
   if (index != axis->numIntervals-1) x2 = axis->intervals[index+1];
   else x2 = axis->max;
   if (x2 == x1) return 0;

   dx = (x2 - x1)/2;
   if (dx < mindx) dx = mindx;
   x1 = minimumPoint - dx;
   x2 = minimumPoint + dx;
   if (x1 < axis->min) x1 = axis->min;
   if (x2 > axis->max) x2 = axis->max;

   dx1 = x1 - axis->min;
   dx2 = axis->max - x2;
   if (dx1 == 0) f1 = 0;
   else f1 = dx1/(dx1+dx2);
   f2 = 1-f1;

   /*  New number of intervals before x1 */
   if (x1 == axis->min) n1 = 0;
   else {
      n1 = (int)(f1*nr + 0.5);
      if (n1 == 0) n1 = 1;
   }
   n2 = nr-n1;

   pind = 0.5;
   p1 = f1*(1 - pind);
   p2 = f2*(1 - pind);

      xc = axis->min;
      if (n1 != 0) {
	axis->intervals[0] = axis->min;
	p = p1/n1;
	axis->prob[0] =p;
      }
      for (i=1; i<n1; i++) {
      	/* axis->intervals[i] = 0.75*xc + 0.25*x1; */
	axis->intervals[i] = 0.5*xc + 0.5*x1; 
      	axis->prob[i] = p;
      	xc = axis->intervals[i];
      }



   /*  Calculating the coordinates and probs of intervals between x1 and x2 */
   dx = (x2 - x1)/nd;
   p = pind/nd;
   for (i=0; i<nd; i++) {
      axis->intervals[n1+i] = x1 + i*dx;
      axis->prob[n1+i] = p;
   }

      xc = axis->max;
      if (n2 != 0) {
	p = p2/n2;
	axis->prob[axis->numIntervals-1] =p;
      }
      for (i=1; i<=n2; i++) {
      	axis->intervals[axis->numIntervals-i] = 0.5*xc + 0.5*x2;
      	axis->prob[axis->numIntervals-i] = p;
      	xc = axis->intervals[axis->numIntervals-i];
      }


   sum = 0;
   for (i=0; i <axis->numIntervals; i++) sum += axis->prob[i];

   for (i=0; i <axis->numIntervals; i++) {
      axis->prob[i] /= sum;
      if (i !=  axis->numIntervals-1) 
	axis->cdf[i+1] = axis->cdf[i] + axis->prob[i];
   } 


   return 1;
}
/*---------------------------------------------------------------------------*/

int updateIntervals(struct ProblemSetup *setup, int iSDC ,double *prevbest, double *amin, double *amax) {
   int i, index;
   double mindx;
   
   for (i=0; i<setup->numVars; i++) {
	index = PAxis_indexOf(setup->axes[i], setup->minimumPoint[i]);
	mindx = 0;
	if (iSDC != 0) mindx = fabs(setup->minimumPoint[i] - prevbest[i]);
	divideInterval(setup->axes[i], index, setup->minimumPoint[i],mindx, amin[i], amax[i]);
   }
   
   return 1;
}
/*---------------------------------------------------------------------------*/


/*  returns a random number between 0.0 and 1.0 */
double randomNumber() {
   
   unsigned int last = RAND_MAX;
   int num = rand();
   return (1.0*num/last);
}
/*---------------------------------------------------------------------------*/
 
void printPoint(int n, double *x) {
   int i;
   for (i=0; i<n; i++) printf("%g\t", x[i]);
   return;
}
/*---------------------------------------------------------------------------*/


/*  This function returns a random number between 0 and 1 with a 
symmetric triangular probability distribution */
double weightedRandomNumber() {
   double t;
   t = randomNumber();

   /*  Added to increase probability of 0.5 
   if (t < 0.5) return 0.5;
   t = (t-0.5)/0.5; */

   if (t < 0.5) return ( sqrt(t/2) );
   return ( 1 - sqrt( 1 - (1+t)/2 ) );
}
/*----------------------------------------------------------------*/

double evaluatePoint(struct ProblemSetup *setup, double *x1) {
	double cost;
    cost = setup->costFunction(setup,x1);
	ProblemSetup_updateMinimum(setup, x1, cost);
	setup->numEvaluations ++;
	return cost;
}


int doFocusingCycle(struct ProblemSetup *setup, double *amin, double *amax) {
   int j;
   double *x1, *prevbest, *genBestPoint, t;
   double cost, prevIterationCost, prevGenCost;
   static int biasCount=0;

   if (setup->costFunction == NULL) fatal_error("Cost function not set up");

   x1 =  (double *)calloc(setup->numVars,sizeof(double) );
   prevbest =  (double *)calloc(setup->numVars,sizeof(double) );
   genBestPoint =  (double *)calloc(setup->numVars,sizeof(double) );
   if (x1  == NULL || prevbest == NULL || genBestPoint == NULL) memory_error();

   for (setup->iFC=0; setup->iFC < setup->NFC; setup->iFC++) {

	  if (setup->numEvaluations > setup->maxEvaluations) break;
      for (setup->iPUC=0; setup->iPUC<setup->NPUC; setup->iPUC++) {
     
		if (setup->numEvaluations > setup->maxEvaluations) break;
      	for (setup->iS=0; setup->iS < setup->NS; setup->iS++) {

		   if (setup->numEvaluations > setup->maxEvaluations) break;
     	   for (j=0; j<setup->numVars; j++) {
				t = randomNumber();
				x1[j] = PAxis_valueForCDF(setup->axes[j], t);
      	   }
      	   cost = evaluatePoint(setup,x1);
		}  /*  end of PUC  */


		if (setup->iPUC == 0 ) copyPoint(setup->numVars, setup->minimumPoint, prevbest);
	   
      	if (!setup->suppressOutput) {
			printf("Iteration %d.%d:\tMinimum=%g\t",setup->iSDC, setup->iPUC, setup->globalMinimum);
			printPoint(setup->numVars, setup->minimumPoint);
			printf("\n");
     	}

    	if (setup->globalMinimum <= setup->threshold) break;

        if (setup->iPUC !=0  && prevIterationCost == setup->globalMinimum) 
			continue;  /* no update */
        prevIterationCost = setup->globalMinimum;
      	updateProbability(setup);

      }  /* end of puc  */
      if (setup->globalMinimum <= setup->threshold) break;
      if (setup->iFC != 0 && prevGenCost == prevIterationCost) 
	   continue; /* no update */

      prevGenCost = prevIterationCost;
      updateIntervals(setup,setup->iSDC ,genBestPoint,amin, amax); /* This order should not be changed */
      copyPoint(setup->numVars, setup->minimumPoint, genBestPoint);

   } /* end  of fc  */
   

   if (x1 != NULL) free(x1);
   if (prevbest != NULL) free(prevbest);
   if (genBestPoint != NULL) free(genBestPoint);

   return 1;
}
/*----------------------------------------------------------------*/
 

double computeWeightedAverage(int axis, int numruns, int numVars, double *points,
	double *costs) {
   int i,worst;
   double sum, x, av;
   double w;

   int ntake = 5;
   if (numruns < ntake) return 0;

   worst = ntake-1;
   sum = 0;
   for (i=0; i<ntake; i++) 
      sum += costs[worst] - costs[i];
      
   av = 0;
   for (i=0; i<ntake; i++) {
      if (sum == 0) w = 0.2;
      else w = (costs[worst] - costs[i])/sum;
      x = points[i*numVars+axis];
      av += x*w;
   }

   return av;
}
/*----------------------------------------------------------------*/

int hasConverged(struct ProblemSetup *setup) {

	int i;
	double dx;

	for(i=0; i<setup->numVars; i++) {
		dx = setup->axes[i]->max - setup->axes[i]->min;
		if (dx > setup->axes[i]->axisPrecision) return 0;	
	}
	return 1;
}
  
int resetAxes(struct ProblemSetup *setup, double *min, double *max) {

	int i;
	double prec;

		for(i=0; i<setup->numVars; i++) {
			prec = setup->axes[i]->axisPrecision;
			PAxis_delete(setup->axes[i]);
			setup->axes[i] = PAxis_create(min[i], max[i]);
			setup->axes[i]->axisPrecision = prec;
		}

		setup->foundmin = 0; 
		return 1;
}

int doRestart(struct ProblemSetup *setup, double *minpoint, double mincost) {
	int savemin=0;

	if (setup->restart != 0) {
		if (setup->backupGlobalMinimum > mincost) savemin=1;
	} else savemin=1;
	if (savemin) {
		setup->backupGlobalMinimum = mincost;
		copyPoint(setup->numVars, minpoint, setup->backupMinimumPoint);
	}
	setup->restart = setup->iSDC;
	return 1;
}
   
int narrowDown(struct ProblemSetup *setup, double *min, double *max, int numruns,
	double *points, double *costs, int bestpoint, int prevbest) {
   int i;
   double xp, x1, x2, dx, av, diff;

   double scf;
   double prec;


   scf = setup->scaleFactor;
   /* if (setup->globalMinimum > costs[bestpoint] ) scf += (0.5 - setup->scaleFactor)*0.9;  */


	if (setup->suppressOutput < 2) printf("Narrowing down\n");
	for(i=0; i<setup->numVars; i++) {
		xp = points[bestpoint*setup->numVars+i];
		dx = setup->axes[i]->max - setup->axes[i]->min;
		dx *= scf;

		if (numruns >= 5) {
			av = computeWeightedAverage(i, numruns, setup->numVars,points,costs);
      		diff = fabs(av - xp)*2;
			if (diff > dx) dx = diff;
		}

		diff = 1.5*fabs( points[ bestpoint*setup->numVars+i] - points[prevbest*setup->numVars+i]);
		if (diff > dx) {
			dx = diff;
		}

		x1 = xp - dx;
		if (x1 < min[i]) {
			x1 = min[i];
			x2 = x1 + 2*dx;
		} else x2 = xp + dx;
		if (x2 > max[i]) x2 = max[i];
		dx = x2 - x1;


		if (setup->suppressOutput < 2) 
			printf("\t%d\t(%g)\t%g\t%g\t%g\n", i, xp, x1, x2, dx);
		prec = setup->axes[i]->axisPrecision;
		PAxis_delete(setup->axes[i]);
		setup->axes[i] = PAxis_create(x1, x2);
		setup->axes[i]->axisPrecision = prec;
	}
	if (setup->suppressOutput < 2) 
		printf("\n");

	setup->foundmin = 0; 
	return 1;
}
/*----------------------------------------------------------------*/


int reinitialiseAxes(struct ProblemSetup *setup) {
   int i;
   double  x1, x2, prec;
   for(i=0; i<setup->numVars; i++) {
      x1 = setup->axes[i]->min;
      x2 = setup->axes[i]->max;
      prec = setup->axes[i]->axisPrecision;
      PAxis_delete(setup->axes[i]);
      setup->axes[i] = PAxis_create(x1, x2);
      setup->axes[i]->axisPrecision = prec;
   }
   setup->foundmin = 0;
   return 1;
}
/*----------------------------------------------------------------*/

/*  Function added in Version 4.  Generates a point in a given direction */
double generatePointAlong(struct ProblemSetup *setup, double *from, double *direction, double step, double *x1) {
	int j;
	double cost;

   for (j=0; j<setup->numVars; j++) {
		x1[j] = from[j] + direction[j]*step;
		if (x1[j] < setup->axes[j]->min)  x1[j] = setup->axes[j]->min;
		if (x1[j] > setup->axes[j]->max)  x1[j] = setup->axes[j]->max;
	}
	cost = 	evaluatePoint(setup, x1);
	return cost;
}

int normalise(struct ProblemSetup *setup, double *vector) {
	int j;
	double sum = 0;
	for (j=0; j<setup->numVars; j++) {
		sum += vector[j]*vector[j];
	}
	sum = sqrt(sum);
	if (fabs(sum) < 1e-100) return 0;
	for (j=0; j<setup->numVars; j++) vector[j] /= sum;
	return 1;

}

int isTooClose(struct ProblemSetup *setup, double *x1, double *x2) {
	int j;
	double dx;
	for (j=0; j<setup->numVars; j++) {
		dx = fabs( x1[j]- x2[j]);
		if (dx > setup->axes[j]->axisPrecision) return 0;
	}
	return 1;
}

/*  Function added in Version 4.  
	Contains two heuristics.  
	1.  Line search along an improving direction
	2.  Testing combinations of solutions
*/
int version4Sampling(struct ProblemSetup *setup, double *amin, double *amax) {
   int j;
   double *x1, *prevbest, dx, dy, *direction, t, step, *v4best;
   double cost, prevGenCost, v4bestCost;
   static int biasCount=0;

   if (setup->costFunction == NULL) fatal_error("Cost function not set up");

   x1 =  (double *)calloc(setup->numVars,sizeof(double) );
   direction =  (double *)calloc(setup->numVars,sizeof(double) );
   prevbest =  (double *)calloc(setup->numVars,sizeof(double) );
   v4best =  (double *)calloc(setup->numVars,sizeof(double) );
   if (x1  == NULL || direction  == NULL || prevbest == NULL || v4best == NULL) memory_error();

   /*  Generating initial point  */
   for (j=0; j<setup->numVars; j++) {
		t = randomNumber();
		x1[j] = PAxis_valueForCDF(setup->axes[j], t);
	}
	cost = 	evaluatePoint(setup, x1);

	copyPoint(setup->numVars, x1, v4best);
	v4bestCost = cost;

	while (1) {

		if (setup->numEvaluations > setup->maxEvaluations) break;
    	if (setup->globalMinimum <= setup->threshold) break;

		copyPoint(setup->numVars, v4best, prevbest);
		prevGenCost = v4bestCost;
		step = 0;
		for (j=0; j < setup->numVars; j++) {
			if (setup->numEvaluations > setup->maxEvaluations) break;
    		if (setup->globalMinimum <= setup->threshold) break;
			copyPoint(setup->numVars, prevbest, x1);
			x1[j] = prevbest[j] + setup->axes[j]->axisPrecision;
			if (x1[j] > setup->axes[j]->max) x1[j] = prevbest[j] - setup->axes[j]->axisPrecision;
			if (x1[j] < setup->axes[j]->min) {
				direction[j] = 0;
				continue;
			}
			cost = 	evaluatePoint(setup, x1);
			dx = x1[j] - prevbest[j];
			dy = cost - prevGenCost;
			if (cost < v4bestCost) {
				copyPoint(setup->numVars, x1, v4best);
				v4bestCost = cost;
			}
			direction[j] = -dy/dx;
			
			if (direction[j] < 0) dx = x1[j] - setup->axes[j]->min;
			else dx = setup->axes[j]->max  - x1[j];
			step += dx*dx/4;
    		if (setup->globalMinimum <= setup->threshold) break;
		}  /*  end of sampling along each axis */

		step = sqrt(step);
		normalise(setup, direction);

		while(1) {
			if (setup->numEvaluations > setup->maxEvaluations) break;
    		if (setup->globalMinimum <= setup->threshold) break;
			cost = generatePointAlong(setup, prevbest, direction, step, x1);
			if (cost < v4bestCost) {
				copyPoint(setup->numVars, x1, v4best);
				v4bestCost = cost;
			}
			if (cost < prevGenCost) break;
			step *= 0.5;
			if (isTooClose(setup, prevbest, x1) ) break;
		}

		if (isTooClose(setup, prevbest, v4best) ) break;

   } /* end  of directional search  */

	if (v4bestCost > setup->globalMinimum) {
		/*  Try combinations of this best with the SDC best  */
		for (j=0; j < setup->numVars; j++) {
			if (setup->numEvaluations > setup->maxEvaluations) break;
    		if (setup->globalMinimum <= setup->threshold) break;
			copyPoint(setup->numVars, setup->minimumPoint, x1);
			dx = fabs(x1[j] -  v4best[j]);
			if (dx < setup->axes[j]->axisPrecision) continue;
			x1[j] = v4best[j];
			cost = 	evaluatePoint(setup, x1);
			if (cost < v4bestCost) {
				copyPoint(setup->numVars, x1, v4best);
				v4bestCost = cost;
			}

		}
	}
   

   if (x1 != NULL) free(x1);
   if (direction != NULL) free(direction);
   if (prevbest != NULL) free(prevbest);
   if (v4best != NULL) free(v4best);

   return 1;
}



int doSubdomainCycle(struct ProblemSetup *setup, double *min, double *max) {
    int i, j, run, bestpoint, prevbest, index, numsaved; 
	double prevbesty;
	int lastimprovement=0;

	if (setup->adaptive) {
		setup->backupMinimumPoint = (double *)calloc(setup->numVars, sizeof(double) );
		if (setup->backupMinimumPoint  == NULL ) memory_error();
	}

    prevbest = 1;
	bestpoint = 0;
    for (setup->iSDC = 0; setup->iSDC < setup->NSDC; setup->iSDC ++) {

		if (setup->numEvaluations > setup->maxEvaluations) break;

		run = setup->iSDC;
		if (setup->restart != 0) run = setup->iSDC-setup->restart-1;

		prevbesty = setup->SDCcosts[bestpoint];
		if (run < numSavedSolutions) version4Sampling(setup,min,max);
		else doFocusingCycle(setup,min,max);
       

		if (setup->suppressOutput < 2) {
			printf("Subdomain %d: Minimum cost = %g\nMinimum Point\n", run, setup->globalMinimum);
			for(i=0; i<setup->numVars; i++) 
					printf("%g ", setup->minimumPoint[i]);
			printf("\n\n");
		}


		if (run == 0) {
			setup->SDCcosts[bestpoint] = setup->globalMinimum;
			copyPoint(setup->numVars, setup->minimumPoint, setup->SDCpoints+bestpoint*setup->numVars);
		} else {
			if (setup->globalMinimum < prevbesty) lastimprovement = run;
			numsaved = 5;
			if (numsaved > run+1) numsaved = run+1;
			index = numsaved-1;
			for (j=0; j<numsaved-1; j++) {
				if (setup->SDCcosts[j] > setup->globalMinimum) {
					index = j;
					break;
				}
			}

			for (j=numsaved-2; j>= index; j--) {
				setup->SDCcosts[j+1] = setup->SDCcosts[j];
				copyPoint(setup->numVars, setup->SDCpoints+j*setup->numVars, setup->SDCpoints+(j+1)*setup->numVars);
			}
			setup->SDCcosts[index] = setup->globalMinimum;
			copyPoint(setup->numVars, setup->minimumPoint, setup->SDCpoints+index*setup->numVars);
		}
		if (setup->globalMinimum <= setup->threshold) break;
		if (run == 0) prevbest = 0;
		else prevbest = 1;

		if (run > 0 && run != setup->NSDC-1) {
			if (hasConverged(setup) || run - lastimprovement > setup->numVars ) {
				lastimprovement = run;
				resetAxes(setup, min, max);
				if (setup->adaptive) doRestart(setup, setup->SDCpoints+bestpoint*setup->numVars, setup->SDCcosts[bestpoint]);
			} else {
				if (run > numSavedSolutions) 
					narrowDown(setup, min, max, run, setup->SDCpoints, setup->SDCcosts, bestpoint, prevbest);
			}
		}
		setup->globalMinimum = setup->SDCcosts[bestpoint];

	}

	if (setup->suppressOutput < 2) {
      printf("Global Minimum cost = %g\nMinimum Point\n", setup->SDCcosts[bestpoint]);
      for(i=0; i<setup->numVars; i++) 
        printf("%g ", setup->SDCpoints[bestpoint*setup->numVars+i]);
      printf("\n\n");
	}

	setup->globalMinimum = setup->SDCcosts[bestpoint];
    copyPoint(setup->numVars, setup->SDCpoints+bestpoint*setup->numVars, setup->minimumPoint);
	if (setup->restart != 0) {
		if (setup->backupGlobalMinimum < setup->globalMinimum) {
			setup->globalMinimum = setup->backupGlobalMinimum;
			copyPoint(setup->numVars, setup->backupMinimumPoint, setup->minimumPoint);
		}
	}


    return 1;
}

double findMinimum(struct ProblemSetup *setup) {

   double *min, *max;
   int i;

   setup->maxEvaluations = setup->NS *setup->NPUC * setup->NFC * setup->NSDC;

   min = (double *)calloc(setup->numVars, sizeof(double) );
   max = (double *)calloc(setup->numVars, sizeof(double) );
   if (min== NULL || max== NULL) memory_error();
   for (i=0; i<setup->numVars; i++) {
      min[i] = setup->axes[i]->min;
      max[i] = setup->axes[i]->max;
   }

   /* Added 20-2-2004  */
   srand(setup->randomSeed); 

   doSubdomainCycle(setup,  min,  max);
   if (min != NULL) free(min);
   if (max != NULL) free(max);
   return setup->globalMinimum;   
}

void copySetupParameters(struct ProblemSetup *setup, struct ProblemSetup *setup2){
                setup2->costFunction = setup->costFunction;
                setup2->NS = setup->NS ;
                setup2->NPUC = setup->NPUC ;
                setup2->NFC = setup->NPUC ;
                setup2->NSDC = setup->NSDC ;
                setup2->iS = setup->NS ;
                setup2->iPUC = setup->NPUC ;
                setup2->iFC = setup->NPUC ;
                setup2->iSDC = setup->NSDC ;
                setup2->threshold = setup->threshold ;
                setup2->randomSeed = setup->randomSeed ;
                setup2->suppressOutput= setup->suppressOutput;
                setup2->scaleFactor = setup->scaleFactor ;
}

