/*  

UserHook.c : User needs to modify only this file to setup a constrained optimisation problem.

Copyright (C) Benny Raphael  

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or  any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.


*/


#include <stdio.h>
#include <math.h>

#include "ConstraintSolver.h"


/* 

  Define the objectiveFunction to minimize. The values of variables are in the array paramValues.

*/
double objectiveFunction(struct ConstraintProblem *setup, double *paramValues){

  double value=0;

  /* Getting the values of parameters from the array.  */
  double x1 = paramValues[0];
  double x2 = paramValues[1];
  double x3 = paramValues[2];

  /*  Calculating the value of the objective function  */
  value = x1*x1 + x2*x2 + x3*x3;

  return value;
}




/* 
  Define a constraint of the form C(x) < 0.  The following function
  evaluates the LHS of the constraint.  The values of variables are in the array paramValues.
  If the function returns a negative or zero value no penalty is added by the PGSL constraint solver.
  Otherwise a penalty is added proportional to the value of this function
*/
double constraint1(struct ConstraintProblem *setup, double *paramValues){

  double value=0;

  /* Getting the values of parameters from the array.  */
  double x1 = paramValues[0];
  double x2 = paramValues[1];
  double x3 = paramValues[2];

  /*  Calculating the value of the objective function  */
  value = x1 + x2 + x3*x3*x3 + 1;

  return value;
}


/* 
  Define another constraint of the form C(x) < 0.  The following function
  evaluates the LHS of the constraint.  The values of variables are in the array paramValues.
*/
double constraint2(struct ConstraintProblem *setup, double *paramValues){

  double value=0;

  /* Getting the values of parameters from the array.  */
  double x1 = paramValues[0];
  double x2 = paramValues[1];
  double x3 = paramValues[2];

  /*  Calculating the value of the objective function  */
  value = x1*x1*x1 + x2 + x3;

  return value;
}



struct ConstraintProblem *setupConstraintProblem() {

	/*  Specify some title.  This appears in the printouts  */
	char *title = "A test problem created by Benny Raphael for illustration";

	double variableLeftBound = -100;	/*  The min bound of variables  */
	double variableRightBound = 100;	/*  The max bound of variables  */

	/*  The maximum number of evaluations of the objective function.  
		Set the value of this variable depending on the amount of time you are prepared to wait to solve the problem.
		Give a large value if you have no idea how long it takes.  You can stop the program in between and still
		see the best solution found so far in the outut file.
	*/
	long maxNumEvaluations=10000L;		

	int numVariables=3;					/*  The number of variables in the problem  */

	int numConstraints=2;				/*  The number of constraints in the problem  */

	/*  Structure representing the constrained optimisation problem  */
	struct ConstraintProblem *problem;
    
	/*  The minimum value of the objective function at which the algorithm
	should terminate.  An impossibly small value could be given if 
	you have no idea of the minimum.
	*/
	double threshold = 1e-8;

	/*  Setting up the problem */
	problem = ConstraintProblem_create(title, numVariables, numConstraints, maxNumEvaluations, threshold);

	/*  Create all the variables  - the arguments are the minimum and maximum values of the variable  - the bounds  
		There should be one PAxis_create statement for each variable.  
		In this problem all variables have the same bounds
	*/
	problem->axes[0] = PAxis_create(variableLeftBound,  variableRightBound);
	problem->axes[1] = PAxis_create(variableLeftBound,  variableRightBound);
	problem->axes[2] = PAxis_create(variableLeftBound,  variableRightBound);


	/*  Specify the pointer to the objective function  */
	problem->objectiveFunction = objectiveFunction;

	/*  Specify the pointer to all the constraints  
		There should be one statement for each constraint.  
	*/
	problem->constraints[0]->evaluationFunction = constraint1;
	problem->constraints[1]->evaluationFunction = constraint2;

	/*  The best solution is written to this file  */
	problem->outputfilename = "constraintsolver.out";

	return problem;

}


/*  Use this function to do any cleaning up or notification.  
	This is called after optimisation and just before deleting the problemsetup.
	You do not need to free any memory allocated inside setup since this is already done by the constraint solver
*/
void doCleanup(struct ConstraintProblem *setup) {
	/*  For this problem there is not cleanup to be performed  */
}
