/*  

ConstraintSolver.c : Use of PGSL to solve constrained problems.

Copyright (C) Benny Raphael  

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or  any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.


*/


#include <stdio.h>
#include <math.h>
#include <malloc.h>
#include <stdlib.h>

/*  Including the PGSL header file  */
#include "../PGSL.h"
#include "ConstraintSolver.h"


struct ConstraintProblem *ConstraintProblem_create(char *title, int numVariables, int numConstraints, long maxNumEvaluations, double threshold) {
	struct ConstraintProblem *ret;
	int i;

	if (numVariables <= 0) {
		printf("The number of variables cannot be less than 1\n");
		return NULL;
	}

	ret = (struct ConstraintProblem *)calloc(1, sizeof(struct ConstraintProblem) );
	if (ret == NULL) memory_error();

	ret->numVars = numVariables;
	ret->title = title;
	ret->maxNumEvaluations = maxNumEvaluations;
	ret->threshold = threshold;
	ret->numConstraints = numConstraints;

	ret->axes = (struct PAxis **)calloc(numVariables, sizeof(struct PAxis *) );
	if (ret->axes == NULL ) memory_error();

	
	if (ret->numConstraints > 0)  {
		ret->constraints = (struct Constraint **)calloc(numConstraints, sizeof(struct Constraint *) );
		ret->penalities = (double *)calloc(numConstraints, sizeof(double) );
		if (ret->constraints  == NULL || ret->penalities == NULL ) memory_error();

		for (i=0; i<numConstraints; i++) {
			ret->constraints[i] = (struct Constraint *)calloc(1, sizeof(struct Constraint) );
			if (ret->constraints[i]  == NULL) memory_error();
			ret->constraints[i]->penaltyFactor = 1e6;
		}

	}

	ret->minimumPoint = (double *)calloc(numVariables, sizeof(double) );
	if (ret->minimumPoint == NULL ) memory_error();

	return ret;
}

void ConstraintProblem_delete(struct ConstraintProblem *problem) {

	int i;

	for (i=0; i<problem->numConstraints; i++) {
		if (problem->constraints[i] != NULL) free(problem->constraints[i]); problem->constraints[i] = NULL;
	}


	if (problem->axes != NULL) free(problem->axes); problem->axes = NULL;
	if (problem->constraints != NULL) free(problem->constraints); problem->constraints = NULL;
	if (problem->penalities != NULL) free(problem->penalities); problem->penalities = NULL;
	if (problem->minimumPoint != NULL) free(problem->minimumPoint); problem->minimumPoint = NULL;

	if (problem->PGSLSetup != NULL) ProblemSetup_delete(problem->PGSLSetup);

}

void saveSolution(struct ConstraintProblem *constraintproblem, double *paramValues, double cost) {

	FILE *out;
	int i;

	printf("\t %ld \t %lg \n", constraintproblem->numEvaluations, cost);

	out = fopen(constraintproblem->outputfilename, "w");
	if (out == NULL) {
		printf("Unable to open %s for writing results.  \n", constraintproblem->outputfilename);
		return;
	}

	fprintf(out, "%s\n", constraintproblem->title);
	fprintf(out, "Trial %ld\n", constraintproblem->numEvaluations);
	fprintf(out, "Best cost so far %lg\n\n\n", cost);
	fprintf(out, "Variable values\n");

	for (i=0; i<constraintproblem->numVars; i++) {
		fprintf(out, "\t%d\t%lg\n", i, paramValues[i]);
	}


	fprintf(out, "\n\n\nConstraint violations\n");
	for (i=0; i<constraintproblem->numConstraints; i++) {
		fprintf(out, "\t%d\t%lg\n", i, constraintproblem->penalities[i]);
	}

	fclose(out);
}


/* 

  The PGSL objectiveFunction to minimize is a combination of the user's objective function 
  and the penalties for constraints.

*/
double combinedObjectiveFunction(struct ProblemSetup *setup, double *paramValues){

	struct ConstraintProblem *constraintproblem;

	double value=0;
	double penalty;
	int i;


	constraintproblem = (struct ConstraintProblem *)(setup->userData);


	/* Updating the number of evaluations  */
	constraintproblem->numEvaluations++;

	value = constraintproblem->objectiveFunction(constraintproblem, paramValues);


	for (i=0; i<constraintproblem->numConstraints; i++) {
		penalty = constraintproblem->constraints[i]->evaluationFunction(constraintproblem, paramValues);
		constraintproblem->penalities[i] = penalty;
		if (penalty > 0) {
			penalty = constraintproblem->constraints[i]->penaltyFactor*(1+penalty);
			value += penalty;
		} else constraintproblem->penalities[i] = 0;
	}

	if (constraintproblem->numEvaluations == 1) {
		constraintproblem->globalMinimum = value;
		copyPoint(setup->numVars, paramValues, constraintproblem->minimumPoint);
	} else {
		if (constraintproblem->globalMinimum > value) {
			constraintproblem->globalMinimum = value;
			copyPoint(setup->numVars, paramValues, constraintproblem->minimumPoint);
			saveSolution(constraintproblem, paramValues, value);
		}
	}

	return value;
}



main() {

	/*  Structure representing the optimisation problem  */
	struct ProblemSetup *setup;

	struct ConstraintProblem *constraintproblem;
   
	int i;


	constraintproblem = setupConstraintProblem();

	if (constraintproblem->objectiveFunction == NULL) {
		printf("The objective function is not specified\n");
		exit (1);
	}

	if (constraintproblem->numConstraints < 0) {
		printf("The number of constraints should be greater than or equal to zero\n");
		exit (1);
	}

	if (constraintproblem->numVars <= 0) {
		printf("The number of variables should be greater than zero\n");
		exit (1);
	}

	for (i=0; i<constraintproblem->numVars; i++) {
		if (constraintproblem->axes == NULL) {
			printf("Axis for variable %d has not been created\n", i);
			exit (1);
		}		
	}

	for (i=0; i<constraintproblem->numConstraints; i++) {
		if (constraintproblem->constraints[i]->evaluationFunction == NULL) {
			printf("The evaluation function for the constraint %d has not been specified\n", i);
			exit (1);
		}		
	}

	if (constraintproblem->outputfilename == NULL) constraintproblem->outputfilename = "constraintsolver.out";

	if (constraintproblem->maxNumEvaluations < 120*constraintproblem->numVars) 
		constraintproblem->maxNumEvaluations = 120*constraintproblem->numVars;

	if (constraintproblem->NFC == 0) constraintproblem->NFC = 20*constraintproblem->numVars;
	constraintproblem->NSDC = constraintproblem->maxNumEvaluations/(40*constraintproblem->numVars);
	
	/*  Setting up the problem */
	setup = ProblemSetup_create(constraintproblem->numVars, constraintproblem->NFC, constraintproblem->NSDC, constraintproblem->threshold);
	setup->costFunction = combinedObjectiveFunction;

	for (i=0; i<constraintproblem->numVars; i++) {
       setup->axes[i] = PAxis_create(constraintproblem->axes[i]->min,  constraintproblem->axes[i]->max);
	   setup->axes[i]->axisPrecision = constraintproblem->axes[i]->axisPrecision;
	}

	setup->userData = constraintproblem;


	/* Calling PGSL  */
	findMinimum(setup);

	doCleanup(constraintproblem);

	ConstraintProblem_delete(constraintproblem);

}


