/*
MATLAB Interface to PGSL

Copyright (C) Benny Raphael  

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or  any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

This file is adapted from the mex interface developed by the COCONUT project team

*/

#include "mex.h"
#include "PGSL.h"
#include <string.h>
#include <stdio.h>

/*  To keep track of the number of evaluations  */
int numTrials = 0;

/*  The left hand side and right hand side list of arguments for calling matlab functions  */
mxArray *rhs[1], *lhs[1];

/*  This is the objective function minimised by PGSL */
double objectiveFunction(struct ProblemSetup *setup, double *paramValues)
{
  char funcname[12] = "objective1";   /*  The name of the objective function in matlab is specified here */
  double value, *paramVal;
  int  numVariables, i;

  numVariables = setup->numVars;
  paramVal = mxGetPr(rhs[0]);
  for(i=0; i<numVariables; i++)
    paramVal[i] = paramValues[i];

  /** Calling the objective function defined in matlab  */
  mexCallMATLAB(1,lhs,1,rhs,funcname);

  numTrials++;
  value = mxGetScalar(lhs[0]);
  return value;
}

/**  This function is called from MATLAB with 7 input parameters.  The input parameters are
0.  An array containing the minimum values of all optimisation variables
1.  An array containing the maximum values of all optimisation variables
2.  An array containing the precision of all optimisation variables
3.  The number of iterations in the focusing cycle NFC
4.  The number of iterations in the subdomain cycle NSDC
5.  The random seed
6.  The threshold of the objective function.  Specify an impossible small value if you do not have an estimate of the threshold
The output parameters are
0.  The best value of the objective function
1.  The best solution found
2.  The number of function evaluations
**/
void mexFunction( int nlhs, mxArray *plhs[], int nrhs, const mxArray *prhs[])
{
  double *minBound, *maxBound, *precision, threshold, *fbest, *xbest, *ncall;
  int NFC, NSDC, randomseed, outfilenum, i, m, n, numVars, numSolutions;
  struct ProblemSetup *setup;

  /**  Checking the number of input variables passed by the matlab function  */
  if(nrhs!=7) mexErrMsgTxt("Seven inputs required");
  if(nlhs>3) mexErrMsgTxt("Too many output arguments");

  minBound = mxGetPr(prhs[0]);
  maxBound = mxGetPr(prhs[1]);
  precision = mxGetPr(prhs[2]);
  NFC = (int)mxGetScalar(prhs[3]);
  NSDC = (int)mxGetScalar(prhs[4]);
  randomseed = (int)mxGetScalar(prhs[5]);
  threshold = (double)mxGetScalar(prhs[6]);
  m = mxGetM(prhs[0]);
  n = mxGetN(prhs[0]);
  if(m>1) numVars = m;
  else numVars = n;

  plhs[0] = mxCreateDoubleMatrix(1, 1, mxREAL);
  plhs[1] = mxCreateDoubleMatrix(m, n, mxREAL);
  plhs[2] = mxCreateDoubleMatrix(1,1, mxREAL);
  fbest = mxGetPr(plhs[0]);
  xbest = mxGetPr(plhs[1]);
  ncall = mxGetPr(plhs[2]);

  setup = ProblemSetup_create(numVars, NFC, NSDC, threshold);
  setup->costFunction = objectiveFunction;
  setup->randomSeed = randomseed;
  for (i=0; i<numVars; i++) {
    setup->axes[i] = PAxis_create(minBound[i], maxBound[i]);
    setup->axes[i]->axisPrecision = precision[i];
  }
  setup->threshold = threshold;
  setup->suppressOutput = 2;
  numTrials = 0;

  rhs[0] = mxCreateDoubleMatrix(numVars,1,mxREAL);
  lhs[0] = mxCreateDoubleMatrix(1,1,mxREAL);

  fbest[0] = findMinimum(setup);

  for (i=0; i<numVars; i++)
    xbest[i] = setup->minimumPoint[i];
  ncall[0] = (int)numTrials;

  mxDestroyArray(rhs[0]);
  mxDestroyArray(lhs[0]);

}

